<?php
/**
 * Plugin Name: Security Headers Manager
 * Plugin URI:  Grimshadow.net/
 * Description: Easy UI to add modern security headers. mostly a fix for jetpack cache breaking security headers. Supports cache-proof PHP prepend (.user.ini) and optional MU-plugin fallback.
 * Version: 1.3.0
 * Author: GrimShadow
 */

if (!defined('ABSPATH')) exit;

class SHC_Plugin {
    const OPT = 'shc_settings';
    const NONCE = 'shc_save';
    const PREPEND_FILENAME = 'shm-prepend.php';          // lives in wp-content/
    const MU_FILENAME = 'security-headers-mu.php';       // lives in wp-content/mu-plugins/
    const USER_INI = '.user.ini';                        // lives in ABSPATH

public static function defaults() {
    return [
        'use_prepend'   => 1,
        'use_mu'        => 0,
        'exclude_admin' => 1, // NEW: skip prepend in /wp-admin and wp-login.php
        'hsts'          => 'max-age=2592000',
        'csp_upgrade'   => 1,
        'xcto'          => 1,
        'referrer'      => 'strict-origin-when-cross-origin',
        'perm'          => 'accelerometer=(), camera=(), geolocation=(), gyroscope=(), magnetometer=(), microphone=(), payment=(), usb=(), bluetooth=(), fullscreen=(self)',
        'xfo'           => 'SAMEORIGIN',
        'xxss'          => '0',
    ];
}

    public static function get_settings() {
        $s = get_option(self::OPT, []);
        return array_merge(self::defaults(), is_array($s) ? $s : []);
    }

    public static function headers_array($https_only = true) {
        $s = self::get_settings();
        $h = [];

        if ($https_only && !empty($s['hsts'])) {
            $h['Strict-Transport-Security'] = $s['hsts'];
        }
        if (!empty($s['csp_upgrade'])) {
            $h['Content-Security-Policy'] = 'upgrade-insecure-requests';
        }
        if (!empty($s['xcto'])) {
            $h['X-Content-Type-Options'] = 'nosniff';
        }
        if (!empty($s['referrer'])) {
            $h['Referrer-Policy'] = $s['referrer'];
        }
        if (!empty($s['perm'])) {
            $h['Permissions-Policy'] = $s['perm'];
        }
        if (!empty($s['xfo'])) {
            $h['X-Frame-Options'] = $s['xfo'];
        }
        if (!empty($s['xxss']) && $s['xxss'] !== '0') {
            $h['X-XSS-Protection'] = $s['xxss'];
        }
        return $h;
    }

    /** ---------- ADMIN UI ---------- */

    public static function admin_menu() {
        add_options_page('Security Headers', 'Security Headers', 'manage_options', 'security-headers', [__CLASS__, 'render_page']);
    }

    public static function render_page() {
        if (!current_user_can('manage_options')) return;

        $msg = '';

        // Handle save
        if (isset($_POST['shc_submit']) && check_admin_referer(self::NONCE)) {
            $new = [
                'exclude_admin' => isset($_POST['exclude_admin']) ? 1 : 0,
                'use_prepend' => isset($_POST['use_prepend']) ? 1 : 0,
                'use_mu'      => isset($_POST['use_mu']) ? 1 : 0,
                'hsts'        => trim(stripslashes($_POST['hsts'] ?? '')),
                'csp_upgrade' => isset($_POST['csp_upgrade']) ? 1 : 0,
                'xcto'        => isset($_POST['xcto']) ? 1 : 0,
                'referrer'    => sanitize_text_field($_POST['referrer'] ?? 'strict-origin-when-cross-origin'),
                'perm'        => trim(stripslashes($_POST['perm'] ?? '')),
                'xfo'         => sanitize_text_field($_POST['xfo'] ?? 'SAMEORIGIN'),
                'xxss'        => ($_POST['xxss'] ?? '0') === '1' ? '1; mode=block' : '0',
            ];
            update_option(self::OPT, $new);
            $msg .= self::apply_choices($new);
        }

        $s = self::get_settings();
        $status = self::status_report();
        $test   = self::loopback_test();

        echo '<div class="wrap"><h1>Security Headers</h1>';
        if (!empty($msg)) echo '<div class="updated"><p>'.esc_html($msg).'</p></div>';
        echo '<form method="post">';
        wp_nonce_field(self::NONCE);

        echo '<h2>Modes</h2><p>';
        echo '<label><input type="checkbox" name="use_prepend" '.checked($s['use_prepend'],1,false).'> Use PHP Prepend (.user.ini) <strong>recommended</strong></label><br>';
        echo '<label style="margin-left:24px"><input type="checkbox" name="exclude_admin" '.checked($s['exclude_admin'],1,false).'> Exclude wp-admin/wp-login from prepend (recommended)</label><br>';

        echo '<label><input type="checkbox" name="use_mu" '.checked($s['use_mu'],1,false).'> Also use MU-plugin fallback (dynamic requests)</label>';
        echo '</p>';

        echo '<h2>Headers</h2>';
        echo '<p>HSTS (HTTPS only): <input type="text" name="hsts" size="60" value="'.esc_attr($s['hsts']).'"> ';
        echo '<em>Example long-term: max-age=31536000; includeSubDomains; preload</em></p>';

        echo '<p><label><input type="checkbox" name="csp_upgrade" '.checked($s['csp_upgrade'],1,false).'> Content-Security-Policy: upgrade-insecure-requests</label></p>';
        echo '<p><label><input type="checkbox" name="xcto" '.checked($s['xcto'],1,false).'> X-Content-Type-Options: nosniff</label></p>';

        $refopts = ['no-referrer','no-referrer-when-downgrade','origin','origin-when-cross-origin','same-origin','strict-origin','strict-origin-when-cross-origin','unsafe-url'];
        echo '<p>Referrer-Policy: <select name="referrer">';
        foreach ($refopts as $opt) printf('<option value="%s"%s>%s</option>', esc_attr($opt), selected($s['referrer'],$opt,false), esc_html($opt));
        echo '</select></p>';

        echo '<p>Permissions-Policy:<br><textarea name="perm" rows="3" style="width:100%;">'.esc_textarea($s['perm']).'</textarea></p>';

        $xfo = ['SAMEORIGIN','DENY','ALLOW-FROM https://example.com'];
        echo '<p>X-Frame-Options: <select name="xfo">';
        foreach ($xfo as $opt) printf('<option value="%s"%s>%s</option>', esc_attr($opt), selected($s['xfo'],$opt,false), esc_html($opt));
        echo '</select></p>';

        echo '<p>X-XSS-Protection: <select name="xxss">';
        echo '<option value="0" '.selected($s['xxss'],'0',false).'>0 (modern default)</option>';
        echo '<option value="1" '.selected($s['xxss'],'1; mode=block',false).'>1; mode=block (for scanners)</option>';
        echo '</select></p>';

        echo '<p><button class="button button-primary" name="shc_submit" value="1">Save & Apply</button></p>';
        echo '</form>';

        echo '<h2>Status</h2><ul style="line-height:1.8">';
        foreach ($status as $k=>$v) echo '<li><strong>'.esc_html($k).':</strong> '.esc_html($v).'</li>';
        echo '</ul>';

        echo '<h2>Live Response Check</h2>';
        if (is_wp_error($test)) {
            echo '<p><em>Loopback test failed: '.esc_html($test->get_error_message()).'</em></p>';
        } else {
            $hdrs = wp_remote_retrieve_headers($test);
            echo '<p>HEAD '.esc_html(home_url('/')).'</p><pre style="max-height:300px;overflow:auto;background:#111;color:#eee;padding:10px;">';
            foreach ($hdrs as $name => $val) {
                if (is_array($val)) $val = implode(', ',$val);
                printf("%s: %s\n", $name, $val);
            }
            echo '</pre>';
        }

        echo '<p><small>Tip: after a week with everything HTTPS (incl. subdomains), switch HSTS to <code>max-age=31536000; includeSubDomains; preload</code>.</small></p>';
        echo '</div>';
    }

    /** ---------- APPLY CHOICES ---------- */

    private static function apply_choices(array $s) {
        $msgs = [];

        // Prepend on/off
        if ($s['use_prepend']) {
            $ok1 = self::write_prepend_file();
            $ok2 = self::write_user_ini(true);
            $msgs[] = ($ok1 && $ok2) ? 'Prepend enabled (.user.ini + prepend.php written).' :
                     'Tried to enable prepend; check file permissions if headers don’t appear.';
        } else {
            $ok = self::write_user_ini(false);
            $msgs[] = $ok ? 'Prepend disabled (.user.ini removed/renamed).' : 'Tried to disable prepend; verify .user.ini removed.';
        }

        // MU on/off
        if ($s['use_mu']) {
            $msgs[] = self::write_mu_plugin() ? 'MU fallback enabled.' : 'Could not write MU fallback (check wp-content/mu-plugins/).';
        } else {
            self::remove_mu_plugin();
            $msgs[] = 'MU fallback disabled.';
        }
        // Ensure admin override .user.ini matches the desired state
        self::write_admin_user_ini( $s['use_prepend'] && $s['exclude_admin'] );

        // If prepend is enabled, re-write prepend file so it includes/excludes admin guard as configured
        if ($s['use_prepend']) self::write_prepend_file();


        return implode(' ', $msgs);
    }

    private static function write_admin_user_ini($enable) {
    $file = ABSPATH . 'wp-admin/.user.ini';
    if ($enable) {
        // Blank the directive so prepend is disabled within /wp-admin
        $content = "auto_prepend_file=\n";
        // Ensure wp-admin exists and is writable
        if (!is_dir(ABSPATH . 'wp-admin/')) return false;
        return (bool) @file_put_contents($file, $content);
    } else {
        if (file_exists($file)) return (bool) @unlink($file);
        return true;
    }
    }


    /** ---------- FILE OPS ---------- */

   private static function write_prepend_file() {
    $path = WP_CONTENT_DIR . '/' . self::PREPEND_FILENAME;
    $h = self::headers_array(true);
    $s = self::get_settings();

    $buf = "<?php\n";

    // Optional admin/login skip (guard runs before anything else)
    if (!empty($s['exclude_admin'])) {
        $buf .= "\$__uri = \$_SERVER['REQUEST_URI'] ?? '';\n";
        $buf .= "\$__script = \$_SERVER['SCRIPT_NAME'] ?? '';\n";
        $buf .= "if (strpos(\$__uri, '/wp-admin/') !== false || strpos(\$__script, 'wp-login.php') !== false) { return; }\n";
    }

    $buf .= "if (!function_exists('sh_set_header_once')) {\n";
    $buf .= "  function sh_set_header_once(\$name, \$value){ if(function_exists('headers_list')){ foreach(headers_list() as \$h){ if(stripos(\$h, \$name.':')===0) return; } } header(\$name.': '.\$value); }\n";
    $buf .= "}\n";
    $buf .= "\$https = (!empty(\$_SERVER['HTTPS']) && \$_SERVER['HTTPS']==='on') || (!empty(\$_SERVER['HTTP_X_FORWARDED_PROTO']) && \$_SERVER['HTTP_X_FORWARDED_PROTO']==='https');\n";

    if (isset($h['Strict-Transport-Security'])) {
        $v = addslashes($h['Strict-Transport-Security']);
        $buf .= "if (\$https) sh_set_header_once('Strict-Transport-Security','{$v}');\n";
        unset($h['Strict-Transport-Security']);
    }
    foreach ($h as $name=>$val) {
        $name = addslashes($name);
        $val  = addslashes($val);
        $buf .= "sh_set_header_once('{$name}','{$val}');\n";
    }

    if (!is_dir(WP_CONTENT_DIR)) return false;
    return (bool) @file_put_contents($path, $buf); 
    }


    private static function write_user_ini($enable) {
        $ini = ABSPATH . self::USER_INI;
        if ($enable) {
            // Use a RELATIVE path to avoid open_basedir issues
            $line = 'auto_prepend_file=wp-content/' . self::PREPEND_FILENAME . "\n";
            return (bool) @file_put_contents($ini, $line);
        } else {
            if (file_exists($ini)) {
                // Try to rename instead of delete (safer)
                $bak = $ini . '.disabled';
                @unlink($bak);
                return (bool) @rename($ini, $bak);
            }
            return true;
        }
    }

    private static function write_mu_plugin() {
    $dir = WP_CONTENT_DIR . '/mu-plugins';
    if (!is_dir($dir)) @wp_mkdir_p($dir);
    if (!is_dir($dir) || !is_writable($dir)) return false;

    $file = $dir . '/' . self::MU_FILENAME;
    $h = self::headers_array(true);
    $s = self::get_settings();

    $buf  = "<?php\n/* Security Headers MU Fallback */\n";
    $buf .= "add_action('send_headers', function(){\n";
    $buf .= "  \$https = (!empty(\$_SERVER['HTTPS']) && \$_SERVER['HTTPS']==='on') || (!empty(\$_SERVER['HTTP_X_FORWARDED_PROTO']) && \$_SERVER['HTTP_X_FORWARDED_PROTO']==='https');\n";

    if (!empty($s['exclude_admin'])) {
        // Limit to wp-admin and login when admin is excluded from prepend
        $buf .= "  \$uri = \$_SERVER['REQUEST_URI'] ?? '';\n";
        $buf .= "  \$scr = \$_SERVER['SCRIPT_NAME'] ?? '';\n";
        $buf .= "  if (strpos(\$uri, '/wp-admin/') === false && strpos(\$scr, 'wp-login.php') === false) return;\n";
    }

    if (isset($h['Strict-Transport-Security'])) {
        $v = addslashes($h['Strict-Transport-Security']);
        $buf .= "  if (\$https) header('Strict-Transport-Security: {$v}');\n";
        unset($h['Strict-Transport-Security']);
    }
    foreach ($h as $name=>$val) {
        $name = addslashes($name);
        $val  = addslashes($val);
        $buf .= "  header('{$name}: {$val}');\n";
    }
    $buf .= "}, 10);\n";

    return (bool) @file_put_contents($file, $buf);
    }

    

    private static function remove_mu_plugin() {
        $file = WP_CONTENT_DIR . '/mu-plugins/' . self::MU_FILENAME;
        if (file_exists($file)) @unlink($file);
    }

    /** ---------- STATUS / TEST ---------- */

    private static function status_report() {
        $s = self::get_settings();
        $prependFile = WP_CONTENT_DIR . '/' . self::PREPEND_FILENAME;
        $userIni     = ABSPATH . self::USER_INI;

        return [
            'Prepend desired' => $s['use_prepend'] ? 'On' : 'Off',
            '.user.ini present' => file_exists($userIni) ? 'Yes' : 'No',
            'prepend.php present' => file_exists($prependFile) ? 'Yes' : 'No',
            'MU fallback desired' => $s['use_mu'] ? 'On' : 'Off',
            'MU file present' => file_exists(WP_CONTENT_DIR . '/mu-plugins/' . self::MU_FILENAME) ? 'Yes' : 'No',
        ];
    }

    private static function loopback_test() {
        // HEAD the homepage via WP HTTP API (server-side)
        $args = ['redirection' => 2, 'timeout' => 10, 'sslverify' => false];
        return wp_remote_head(home_url('/'), $args);
    }

    /** ---------- HOOKS ---------- */

    public static function on_activate() {
        // Apply current choices on activation
        self::apply_choices(self::get_settings());
    }
}

add_action('admin_menu', [SHC_Plugin::class, 'admin_menu']);
register_activation_hook(__FILE__, [SHC_Plugin::class, 'on_activate']);
